/*
* mantra.cpp - Faxanadu Password Generator
*
* Copyright (c) 2005 Sebastian Porst (webmaster@the-interweb.com)
* All rights reserved.
*
* This software is licensed under the zlib/libpng License.
* For more details see http://www.opensource.org/licenses/zlib-license.php
*/

#include <iostream>

#include "neshelp.h"

enum Town {EILOS, APOLUNE, FOREPAW, MASCON, VICTIM, CONFLATE, DAYBREAK, DARTMOOR};
enum Title {NOVICE, ASPIRANT, BATTLER, FIGHTER, ADEPT, CHEVALIER, VETERAN,
            WARRIOR, SWORDMAN, HERO, SOLDIER, MYRMIDON, CHAMPION, SUPERHERO, PALADIN, LORD};

enum UnselectableItem {
	NO_ITEMS		= 0x00,
	BLACK_ONYX 		= 0x01,
	PENDANT 		= 0x02,
	MAGICAL_ROD 	= 0x04,
	ELIXIR 			= 0x08,
	DEMONS_RING 	= 0x10,
	RING_OF_DWORF	= 0x20,
	RING_OF_RUBY	= 0x40,
	RING_OF_ELF		= 0x80};
	
enum Quest {
	NO_QUESTS		= 0x00
};

enum Weapon {NO_WEAPON = -1, HAND_DAGGER = 0, LONG_SWORD, GIANT_BLADE, DRAGON_SLAYER};
enum Armor {NO_ARMOR = -1, LEATHER_ARMOR = 0, STUDDED_MAIL, FULL_PLATE, BATTLE_SUIT};

enum Shield {NO_SHIELD = -1, SMALL_SHIELD = 0, LARGE_SHIELD, MAGIC_SHIELD, BATTLE_HELMET};

enum Magic {NO_MAGIC = -1, DELUGE = 0, THUNDER, FIRE, DEATH, TILTE};

enum Item {
	NO_ITEM 			= -1,
	RING_OF_ELF_SEL 	= 0,
	RING_OF_RUBY_SEL,
	RING_OF_DWORF_SEL,
	DEMONS_RING_SEL,
	KEY_A,
	KEY_K,
	KEY_Q,
	KEY_J,
	KEY_JO,
	MATTOCK,
	MAGICAL_ROD_SEL,
	CRYSTAL,
	LAMP,
	HOUR_GLASS,
	BOOK,
	WING_BOOTS,
	RED_POTION,
	ELIXIR_SEL,
	PENDANT_SEL,
	BLACK_ONIX,
	FIRE_CRYSTAL
};

struct Mantra
{
	Town town;
	Title title;
	UnselectableItem uitems;
	Quest quests;
	Weapon selWeapon;
	Armor selArmor;
	Shield selShield;
	Magic selMagic;
	Item selItem;
	Weapon invWeapons[4];
	Armor invArmor[4];
	Shield invShields[4];
	Magic invMagic[4];
	Item invItems[8];
	
	Mantra()
	{
		town = EILOS;
		title = NOVICE;
		uitems = NO_ITEMS;
		quests = NO_QUESTS;
		selWeapon = NO_WEAPON;
		selArmor = NO_ARMOR;
		selShield = NO_SHIELD;
		selMagic = NO_MAGIC;
		selItem = NO_ITEM;
		
		for (unsigned int i=0;i<4;i++) invWeapons[i] = NO_WEAPON;
		for (unsigned int i=0;i<4;i++) invArmor[i] = NO_ARMOR;
		for (unsigned int i=0;i<4;i++) invShields[i] = NO_SHIELD;
		for (unsigned int i=0;i<4;i++) invMagic[i] = NO_MAGIC;
		for (unsigned int i=0;i<8;i++) invItems[i] = NO_ITEM;
	}
};

std::string generateMantra(unsigned char* buffer, unsigned int size)
{
	std::string str;

	for (unsigned int bitpos = 0;bitpos<size;bitpos+=6)
	{
		unsigned int bits = (unsigned int) getBits(buffer, bitpos / 8, bitpos % 8, 6);

		if (bits <= 0x19) str += 'A' + bits;
		else if (bits <= 0x33) str += 'a' + bits - 0x1A;
		else if (bits <= 0x3D) str += '0' + bits - 0x34;
		else if (bits == 0x3E) str += ',';
		else if (bits == 0x3F) str += '?';
	}
	
	return str;
}

int main()
{
	Mantra m;
	m.town = EILOS;
	m.title = LORD;
	m.uitems = (UnselectableItem)(BLACK_ONYX | PENDANT | MAGICAL_ROD | ELIXIR | DEMONS_RING | RING_OF_DWORF | RING_OF_RUBY | RING_OF_ELF);
	m.quests = (Quest)0xFF;
	m.selWeapon = DRAGON_SLAYER;
	m.selArmor = BATTLE_SUIT;
	m.selShield = BATTLE_HELMET;
	m.selMagic = TILTE;
	m.selItem = WING_BOOTS;
	m.invWeapons[0] = HAND_DAGGER;
	m.invWeapons[1] = LONG_SWORD;
	m.invWeapons[2] = GIANT_BLADE;
	m.invWeapons[3] = DRAGON_SLAYER;
	m.invArmor[0] = LEATHER_ARMOR;
	m.invArmor[1] = STUDDED_MAIL;
	m.invArmor[2] = FULL_PLATE;
	m.invArmor[3] = BATTLE_SUIT;
	m.invShields[0] = SMALL_SHIELD;
	m.invShields[1] = LARGE_SHIELD;
	m.invShields[2] = MAGIC_SHIELD;
	m.invShields[3] = BATTLE_HELMET;
	m.invMagic[0] = DELUGE;
	m.invMagic[1] = THUNDER;
	m.invMagic[2] = FIRE;
	m.invMagic[3] = DEATH;
	m.invItems[0] = RED_POTION;
	m.invItems[1] = KEY_J;
	m.invItems[2] = KEY_Q;
	m.invItems[3] = KEY_K;
	m.invItems[4] = KEY_A;
	m.invItems[5] = KEY_JO;
	m.invItems[6] = MATTOCK;
	m.invItems[7] = WING_BOOTS;

	StuffBits<0x20> sb;
	sb.addBits(0, 8);
	sb.addBits(0, 5);
	sb.addBits(m.town, 3);
	sb.addBits(m.title, 4);
	sb.addBits(m.uitems, 8);
	sb.addBits(m.quests, 8);

	sb.addBits(m.selWeapon != NO_WEAPON, 1);
	if (m.selWeapon != NO_WEAPON) sb.addBits(m.selWeapon, 2);

	sb.addBits(m.selArmor != NO_ARMOR, 1);
	if (m.selArmor != NO_ARMOR) sb.addBits(m.selArmor, 2);

	sb.addBits(m.selShield != NO_SHIELD, 1);
	if (m.selShield != NO_SHIELD) sb.addBits(m.selShield, 2);

	sb.addBits(m.selMagic != NO_MAGIC, 1);
	if (m.selMagic != NO_MAGIC) sb.addBits(m.selMagic, 3);

	sb.addBits(m.selItem != NO_ITEM, 1);
	if (m.selItem != NO_ITEM) sb.addBits(m.selItem, 5);

	unsigned int invWeapons = 0;
	for (unsigned int i=0;i<4;i++) if (m.invWeapons[i] != NO_WEAPON) invWeapons++;
	
	sb.addBits(invWeapons, 3);
	
	for (unsigned int i=0;i<4;i++)
	{
		if (m.invWeapons[i] != NO_WEAPON) sb.addBits(m.invWeapons[i], 2);
	}
	
	unsigned int invArmor = 0;
	for (unsigned int i=0;i<4;i++) if (m.invArmor[i] != NO_ARMOR) invArmor++;
	
	sb.addBits(invArmor, 3);
	
	for (unsigned int i=0;i<4;i++)
	{
		if (m.invArmor[i] != NO_ARMOR) sb.addBits(m.invArmor[i], 2);
	}

	unsigned int invShields = 0;
	for (unsigned int i=0;i<4;i++) if (m.invShields[i] != NO_SHIELD) invShields++;
	
	sb.addBits(invShields, 3);
	
	for (unsigned int i=0;i<4;i++)
	{
		if (m.invShields[i] != NO_SHIELD) sb.addBits(m.invShields[i], 2);
	}

	unsigned int invMagic = 0;
	for (unsigned int i=0;i<4;i++) if (m.invMagic[i] != NO_MAGIC) invMagic++;
	
	sb.addBits(invMagic, 3);
	
	for (unsigned int i=0;i<4;i++)
	{
		if (m.invMagic[i] != NO_MAGIC) sb.addBits(m.invMagic[i], 3);
	}

	unsigned int invItems = 0;
	for (unsigned int i=0;i<8;i++) if (m.invItems[i] != NO_ITEM) invItems++;
	
	sb.addBits(invItems, 4);
	
	for (unsigned int i=0;i<8;i++)
	{
		if (m.invItems[i] != NO_ITEM) sb.addBits(m.invItems[i], 5);
	}

	unsigned char size = sb.size();
	
	sb.getBuffer()[1] |= (size / 6 + (size % 6 != 0)) << 3;
	
	unsigned char sum = 0;
	for (unsigned int i=0;i<(unsigned int)size/6;i++) sum += sb.getBuffer()[i];
	
	sb.getBuffer()[0] = -sum;
	
	std::cout << generateMantra(sb.getBuffer(), size) << std::endl;
}
